library(tidyverse)
library(testthat)
library(ggbrace)
library(scales)
library(xlsx)

setwd("~/repo/one_offs")
source("../lab_code/prelim.R")


##### Function to generate stream of payments and UPB from mortgage terms #######
pmt_stream <- function(base_term) {
  # Calculates payments and upbs for a mortgage given terms
  #
  # Args:
  #   base_term: List of:
  #     r_temp: interest rate for first 5 years; fraction
  #     r_perm: interest rate for rest of mortgage; fraction,
  #     term: length of mortgage in years; integer
  #     upb_0: initial unpaid balance in dollars
  #     forbear: amount of forbearance in dollars
  #
  # Returns:
  #   list of:
  #     pay: length 40 stream of payments with padded zeros if necessary
  #     upb: length 40 stream of upb with padded zeros if necessary
  #
  #
  schedule <- data.frame(yrsPaid=c(1:base_term$term),
                         yrsLeft=c(base_term$term:1),
                         r=NA, pay=NA,
                         int_pmt_amt=NA, princ_pmt_amt=NA,
                         upb_beg=NA, upb_end=NA) %>%
    #Set first UPB value from base_term (subtract forbearance amount)
    mutate(upb_beg=ifelse(yrsPaid==1, base_term$upb_0-base_term$forbear, upb_beg))
  
  #Define a vector of interest rates
  for (year in c(1:min(base_term$term, 5))) {
    schedule$r[year] <- base_term$r_temp
  }
  #Increase by 1% until it reaches r_perm
  if (base_term$term > 5) {
    for (year in c(6:base_term$term)) {
      schedule$r[year] <- min(schedule$r[year-1]+0.01, base_term$r_perm)
    }
  }
  
  #Fill up dataframe year by year
  pay_ann <- function(upb, term, r){ #annuity formula for fixed rate, annual term
    return((r*upb)/(1-((1+r)^-term)))
  }
  
  for(i in 1:base_term$term){
    row <- schedule[i,]
    
    row$int_pmt_amt <- row$r*row$upb_beg
    row$pay <- pay_ann(row$upb_beg, row$yrsLeft, row$r)
    row$princ_pmt_amt <- row$pay-row$int_pmt_amt
    row$upb_end <- row$upb_beg - row$princ_pmt_amt
    schedule[i,] <- row
    
    #Update upb_beg for next row
    if(i<base_term$term){
      schedule[i+1,'upb_beg'] <- row$upb_end
    }
  }
  
  
  #Add back forbearance amount and keep only relevant variables
  schedule <- schedule %>%
    mutate(pay=ifelse(yrsPaid==base_term$term, pay+base_term$forbear, pay),
           upb_end=ifelse(yrsPaid!=base_term$term, upb_end+base_term$forbear, upb_end)) %>%
    select(pay, upb_end)
  
  #Pad extra zeros
  if (nrow(schedule) < 40) {
    add_years <- 40-nrow(schedule)
    pad_zeros <- rep(0, add_years)
    schedule_pad <- data.frame(pay=pad_zeros, upb_end=pad_zeros)
    
    schedule <-  bind_rows(schedule, schedule_pad)
  }
  
  payment <- list('pay'=schedule$pay,
                  'upb'=schedule$upb_end)
  
  return(payment)
}

# CARES intervention
pmt_stream_cares <- function(base_term, one_time) {
  # Calculates payments and upbs for a mortgage given terms
  #
  # Args:
  #   base_term: List of:
  #     r: interest rate for rest of mortgage; fraction,
  #     term: length of mortgage in years; integer
  #     upb_0: initial unpaid balance in dollars
  #     forbear: amount of forbearance in dollars (zero for CARES act)
  #
  # Returns:
  #   list of:
  #     pay: length 40 stream of payments with padded zeros if necessary
  #     upb: length 40 stream of upb with padded zeros if necessary
  #
  #
  schedule <- data.frame(yrsPaid=c(1:base_term$term),
                         yrsLeft=c(base_term$term:1),
                         r=NA, pay=NA,
                         int_pmt_amt=NA, princ_pmt_amt=NA,
                         upb_beg=NA, upb_end=NA) %>%
    #Set first UPB value from base_term (subtract forbearance amount)
    mutate(upb_beg=ifelse(yrsPaid==1,
                          base_term$upb_0-base_term$forbear,
                          upb_beg),
           r = base_term$r_perm)
  
  #Fill up dataframe year by year
  pay_ann <- function(upb, term, r){ #annuity formula for fixed rate, annual term
    return((r*upb)/(1-((1+r)^-term)))
  }
  
  for(i in 1:base_term$term){
    row <- schedule[i,]
    
    row$int_pmt_amt <- row$r*row$upb_beg
    row$pay <- pay_ann(row$upb_beg, row$yrsLeft, row$r)
    row$princ_pmt_amt <- row$pay-row$int_pmt_amt
    row$upb_end <- row$upb_beg - row$princ_pmt_amt
    schedule[i,] <- row
    
    #Update upb_beg for next row
    if(i<base_term$term){
      schedule[i+1,'upb_beg'] <- row$upb_end
    }
  }
  
  #Add one-time payment and keep only relevant variables
  if(one_time == "end") {
    schedule <- schedule %>%
      mutate(pay = case_when(yrsPaid == 1 ~ 0,
                             yrsPaid == base_term$term ~ pay * 2,
                             TRUE ~ pay),
             upb_end=ifelse(yrsPaid != base_term$term,
                            upb_end+base_term$forbear,
                            upb_end)) %>%
      select(pay, upb_end)
  }
  
  if(one_time == "start") {
    schedule <- schedule %>%
      mutate(pay = case_when(yrsPaid == 1 ~ 0,
                             yrsPaid == 2 ~ pay * 2,
                             TRUE ~ pay),
             upb_end=ifelse(yrsPaid != base_term$term,
                            upb_end+base_term$forbear,
                            upb_end)) %>%
      select(pay, upb_end)
  }
  
  #Pad extra zeros
  if (nrow(schedule) < 40) {
    add_years <- 40-nrow(schedule)
    pad_zeros <- rep(0, add_years)
    schedule_pad <- data.frame(pay=pad_zeros, upb_end=pad_zeros)
    
    schedule <-  bind_rows(schedule, schedule_pad)
  }
  
  
  payment <- list('pay'=schedule$pay,
                  'upb'=schedule$upb_end)
  
  return(payment)
}

cares_recap <- function(base_term) {
  pre_payments <- pmt_stream(base_term)$pay
  
  first_payment <- pre_payments[1]
  
  forbearance_mortg <- list(r_temp = base_term$r_temp, #pre intervention interest rate first five years
                            r_perm = base_term$r_perm, #same for the period after five years
                            term = (base_term$term - 1), #remaining term
                            upb_0 = first_payment, #unpaid principle balance
                            forbear = 0) #forbearance amount
  
  pre_payments[1] <- 0
  
  post_payments <- pre_payments + data.table::shift(pmt_stream(forbearance_mortg)$pay, fill = 0,  type = "lag")
  
  payment <- list('pay'= post_payments,
                  'upb'= NA)
  
  return(payment)
}

#######################################
# Cash Flow from borrower perspective #
#######################################

# Inputs:
income <- 55596         #income
term_pre_mod <- 26      #pre modification remaining term in year
r_pre_mod <- 0.044       #pre modification interest rate
home_value <- 205275    #home value
ltv <- 1.43             #loan to value ratio
term_length <- 23       #years to maturity
#upb <- home_value * ltv #principal balance
upb <- 223185

pre_mortg <- list(r_temp = r_pre_mod, #pre modification interest rate first five years
                  r_perm = r_pre_mod, #same for the period after five years
                  term = term_length, #remaining term
                  upb_0 = upb, #unpaid principle balance
                  forbear = 0) #forbearance amount

# Calibration of mortgages to match
# payment reduction of {5%, 15%, 25%, 35%, 45%, 55%, 65%}:
mean(pmt_stream(pre_mortg)$pay[1:5]) #initial payment level

pay_goal <- function(x) {
  mean(pmt_stream(pre_mortg)$pay[1:5]) * (1 - x)
}

pay_mean <- function(mortg) {
  mean(pmt_stream(mortg)$pay[1:5])
}

# 5%
mortg_05 <- list(r_temp = 0.0386, #post modification interest rate first five years
                 r_perm = 0.0386, #post modification interest rate after five years
                 term = 23, #remaining term
                 upb_0 = upb, #unpaid principle balance
                 forbear = 0) #forbearance amount

test_that("Meets payment reduction of 5%",
          expect_equal(pay_goal(0.05),
                       pay_mean(mortg_05),
                       tolerance = 0.01))


# 15%
mortg_15 <- list(r_temp = 0.0286, #post modification interest rate first five years
                 r_perm = 0.03, #post modification interest rate after five years
                 term = 23, #remaining term
                 upb_0 = upb, #unpaid principle balance
                 forbear = 0) #forbearance amount

test_that("Meets payment reduction of 15%",
          expect_equal(pay_goal(0.15),
                       pay_mean(mortg_15),
                       tolerance = 0.01))

# 25%
mortg_25 <- list(r_temp = 0.02, #post modification interest rate first five years
                 r_perm = 0.03, #post modification interest rate after five years
                 term = 23, #remaining term
                 upb_0 = upb, #unpaid principle balance
                 forbear = 12500) #forbearance amount
pay_goal(0.25)
pay_mean(mortg_25)

test_that("Meets payment reduction of 25%",
          expect_equal(pay_goal(0.25),
                       pay_mean(mortg_25),
                       tolerance = 0.05)) #less precise

# 35%
mortg_35 <- list(r_temp = 0.0336, #post modification interest rate first five years
                 r_perm = 0.0336, #post modification interest rate after five years
                 term = 40, #remaining term
                 upb_0 = upb, #unpaid principle balance
                 forbear = 0) #forbearance amount

test_that("Meets payment reduction of 35%",
          expect_equal(pay_goal(0.35),
                       pay_mean(mortg_35),
                       tolerance = 100))

# 45%
mortg_45 <- list(r_temp = 0.0236, #post modification interest rate first five years
                 r_perm = 0.03, #post modification interest rate after five years
                 term = 40, #remaining term
                 upb_0 = upb, #unpaid principle balance
                 forbear = 0) #forbearance amount

test_that("Meets payment reduction of 45%",
          expect_equal(pay_goal(0.45),
                       pay_mean(mortg_45),
                       tolerance = 0.05))

# 55%
mortg_55 <- list(r_temp = 0.02, #post modification interest rate first five years
                 r_perm = 0.03, #post modification interest rate after five years
                 term = 40, #remaining term
                 upb_0 = upb, #unpaid principle balance
                 forbear = 31000) #forbearance amount

test_that("Meets payment reduction of 55%",
          expect_equal(pay_goal(0.55),
                       pay_mean(mortg_55),
                       tolerance = 0.01))

# 65%
mortg_65 <- list(r_temp = 0.02, #post modification interest rate first five years
                 r_perm = 0.03, #post modification interest rate after five years
                 term = 40, #remaining term
                 upb_0 = upb, #unpaid principle balance
                 forbear = 75000) #forbearance amount

test_that("Meets payment reduction of 65%",
          expect_equal(pay_goal(0.65),
                       pay_mean(mortg_65),
                       tolerance = 0.01))

# Get average difference for all payment reductions
mortg <- list(mortg_05, mortg_15, mortg_25,
              mortg_35, mortg_45, mortg_55, mortg_65)

eval_mortg <- function(mortg_) {
  tibble(
    year = seq(1, 40, by = 1),
    pre_pay = pmt_stream(pre_mortg)$pay,
    hamp_pay  = pmt_stream(mortg_)$pay,
  ) %>%
    mutate(d_hamp = hamp_pay - pre_pay,
           reduction = 1 - (pay_mean(mortg_) / pay_mean(pre_mortg)))
}

df_hamp <- mortg %>%
  map_dfr(~eval_mortg(.)) %>%
  group_by(year) %>%
  summarise_all(mean) %>%
  select(year, d_hamp)

df_pay <- tibble(
  year = seq(1, 40, by = 1),
  pre_pay = pmt_stream(pre_mortg)$pay,
  cares_pay = pmt_stream_cares(pre_mortg, one_time = "end")$pay
  ) %>%
  transmute(year,
            d_cares = cares_pay - pre_pay) %>%
  left_join(df_hamp, by = "year")

#############################
# Payment Differences Plots #
#############################

# animated HAMP modification
df_pay %>%
  select(year, d_hamp) %>%
  ggplot(aes(x = year, y = d_hamp)) +
  geom_hline(yintercept = 0,
             linetype = "dashed",
             size = 0.25) +
  geom_vline(xintercept = 23,
             linetype = "dashed",
             size = 0.3) +
  annotate("text", x = 22.9, y = 19600,
           label = "Original term length of 23 years",
           hjust = 1, size = 3, family = "serif") +
  geom_col(width = 0.5, fill = "#8ecae6") +
  geom_brace(
    aes(c(1, 5), c(2500, 3000),
        label = "Step 1:\nTemporary low\ninterest rate\n"),
    inherit.data = F,
    family = "serif",
    labelsize = 3) +
  geom_brace(
    aes(c(6, 22), c(2500, 3000),
        label = "Step 2:\nMarket interest rate\n"),
    inherit.data = F,
    family = "serif",
    labelsize = 3) +
  geom_brace(aes(c(24, 40), c(-1000, -1500),
                 label = "\nStep 3:\nterm extension"),
             inherit.data=F, labelsize=3,
             family = "serif",
             rotate = 180) +
  annotate("text", x = 25, y = 15300,
           label = "Step 4: principal forbearance",
           family = "serif",
           size = 3, hjust = 0) +
  annotate("segment", x = 34, xend = 39.5, y = 15500, yend = 19800,
           color = "black") +
  scale_y_continuous(labels = scales::dollar_format(),
                   breaks = seq(-10000, 20000, by = 5000)) +
  scale_x_continuous(limits = c(0, 41),
                     expand = c(0, 0)) +
  labs(x = "Years since intervention",
       y = "",
       subtitle = "Change in annual payment after HAMP for typical mortgage",
       color = "") +
  fte_theme() +
  theme(legend.position = c(0.95, 0.05),
        legend.justification = c("right", "bottom"),
        axis.text = element_text(size = 10),
        plot.title.position = "plot")

ggsave(str_c("issue_22/", "hamp_mod_5_5.pdf"),
       height = 4.5, width = 8, unit = "in")


# both interventions
df_pay %>%
  select(year, d_hamp, d_cares) %>%
  pivot_longer(cols = -year,
               values_to = "d_pay",
               names_to = "mod") %>%
  mutate(mod = factor(mod, c("d_hamp", "d_cares"))) %>%
  ggplot(aes(x = year, y = d_pay, fill = mod)) +
  geom_vline(xintercept = 23,
             linetype = "dashed",
             size = 0.3) +
  annotate("text", x = 23.1, y = -15000,
           label = "Original term length of 23 years",
           family = "serif",
           hjust = 0, size = 3) +
  geom_hline(yintercept = 0,
             linetype = "dashed",
             size = 0.25) +
  geom_col(position = "dodge") +
  scale_y_continuous(labels = scales::dollar_format()) +
  scale_x_continuous(limits = c(0, 41),
                     expand = c(0, 0)) +
  scale_fill_manual(values = c("#8ecae6", "#9d0208"),
                     labels = c("HAMP (Great Recession)",
                                "CARES (Pandemic)")) +
  labs(x = "Years since intervention",
       y = "",
       subtitle = "Change in annual payment after intervention for typical mortgage",
       fill = "") +
  fte_theme() +
  theme(legend.position = c(0.05, 0.95),
        axis.text = element_text(size = 10),
        legend.text = element_text(size = 12),
        legend.justification = c("left", "top"),
        plot.title.position = "plot")

ggsave(str_c("issue_22/", "both_mod_cares_last.pdf"),
       height = 4.5, width = 8, unit = "in")


# df_pay %>%
#   select(year, d_hamp, d_cares) %>%
#   pivot_longer(cols = -year,
#                values_to = "d_pay",
#                names_to = "mod") %>%
#   mutate(mod = factor(mod, c("d_hamp", "d_cares"))) %>%
#   ggplot(aes(x = year, y = d_pay, fill = mod)) +
#   geom_vline(xintercept = 23,
#              linetype = "dashed",
#              size = 0.3) +
#   annotate("text", x = 23.1, y = -15000,
#            label = "Original term length of 23 years",
#            family = "serif",
#            hjust = 0, size = 3) +
#   geom_hline(yintercept = 0,
#              linetype = "dashed",
#              size = 0.25) +
#   geom_col(position = "dodge") +
#   annotate("text", x = 1.7, y = -10000,
#            label = "Year 1 payment reduction: $15,600 (CARES) vs $5,500 (HAMP)\nNPV total payment reduction: $7,300 (CARES) vs  $44,800 (HAMP)",
#            family = "serif",
#            size = 3, hjust = 0) +
#   scale_y_continuous(labels = scales::dollar_format()) +
#   scale_x_continuous(limits = c(0, 41),
#                      expand = c(0, 0)) +
#   scale_fill_manual(values = c("#8ecae6", "#9d0208"),
#                     labels = c("HAMP (Great Recession)",
#                                "CARES (Pandemic)")) +
#   labs(x = "Years since intervention",
#        y = "",
#        subtitle = "Change in annual payment after intervention for typical mortgage",
#        fill = "") +
#   fte_theme() +
#   theme(legend.position = c(0.05, 0.95),
#         axis.text = element_text(size = 10),
#         legend.text = element_text(size = 12),
#         legend.justification = c("left", "top"),
#         plot.title.position = "plot")
# 
# ggsave(str_c("issue_22/", "both_mod_cares_last_annotated.png"),
#        height = 4.5, width = 8, unit = "in")



# tibble(
#   year = seq(1, 40, by = 1),
#   pre_pay = pmt_stream(pre_mortg)$pay,
#   cares_pay = pmt_stream_cares(pre_mortg, one_time = "start")$pay
#   ) %>%
#   transmute(year,
#             d_cares = cares_pay - pre_pay) %>%
#   left_join(df_hamp, by = "year") %>%
#   select(year, d_hamp, d_cares) %>%
#   pivot_longer(cols = -year,
#                values_to = "d_pay",
#                names_to = "mod") %>%
#   mutate(mod = factor(mod, c("d_hamp", "d_cares"))) %>%
#   ggplot(aes(x = year, y = d_pay, fill = mod)) +
#   geom_vline(xintercept = 23,
#              linetype = "dashed",
#              size = 0.3) +
#   annotate("text", x = 23.1, y = 13000,
#            label = "Original term length of 23 years",
#            family = "serif",
#            hjust = 0, size = 3) +
#   geom_hline(yintercept = 0,
#              linetype = "dashed",
#              size = 0.25) +
#   geom_col(position = "dodge") +
#   scale_y_continuous(labels = scales::dollar_format()) +
#   scale_x_continuous(limits = c(0, 41),
#                      expand = c(0, 0)) +
#   scale_fill_manual(values = c("#8ecae6", "#9d0208"),
#                      labels = c("HAMP (Great Recession)",
#                                 "CARES (Pandemic)")) +
#   labs(x = "Years since intervention",
#        y = "",
#        subtitle = "Change in annual payment after intervention for typical mortgage",
#        fill = "") +
#   fte_theme() +
#   theme(legend.position = c(0.95, 0.05),
#         axis.text = element_text(size = 10),
#         legend.text = element_text(size = 12),
#         legend.justification = c("right", "bottom"),
#         plot.title.position = "plot")
# 
# ggsave(str_c("issue_22/", "both_mod_cares_second.png"),
#        height = 4.5, width = 8, unit = "in")


tibble(
  year = seq(1, 40, by = 1),
  pre_pay = pmt_stream(pre_mortg)$pay,
  cares_pay = pmt_stream_cares(pre_mortg, one_time = "start")$pay
) %>%
  transmute(year,
            d_cares = cares_pay - pre_pay) %>%
  left_join(df_hamp, by = "year") %>%
  select(year, d_hamp, d_cares) %>%
  pivot_longer(cols = -year,
               values_to = "d_pay",
               names_to = "mod") %>%
  mutate(mod = factor(mod, c("d_hamp", "d_cares"))) %>%
  ggplot(aes(x = year, y = d_pay, fill = mod)) +
  geom_vline(xintercept = 23,
             linetype = "dashed",
             size = 0.3) +
  annotate("text", x = 23.1, y = 13000,
           label = "Original term length of 23 years",
           family = "serif",
           hjust = 0, size = 3) +
  geom_hline(yintercept = 0,
             linetype = "dashed",
             size = 0.25) +
  geom_col(position = "dodge") +
  annotate("text", x = 1.7, y = -10000,
           label = "Year 1 payment reduction: $15,600 (CARES) vs $5,500 (HAMP)\nNPV total payment reduction: $400 (CARES) vs  $44,800 (HAMP)",
           family = "serif",
           size = 3, hjust = 0) +
  scale_y_continuous(labels = scales::dollar_format()) +
  scale_x_continuous(limits = c(0, 41),
                     expand = c(0, 0)) +
  scale_fill_manual(values = c("#8ecae6", "#9d0208"),
                    labels = c("HAMP (Great Recession)",
                               "CARES (Pandemic)")) +
  labs(x = "Years since intervention",
       y = "",
       subtitle = "Change in annual payment after intervention for typical mortgage",
       fill = "") +
  fte_theme() +
  theme(legend.position = c(0.95, 0.05),
        axis.text = element_text(size = 10),
        legend.text = element_text(size = 12),
        legend.justification = c("right", "bottom"),
        plot.title.position = "plot")

ggsave(str_c("issue_22/", "both_mod_cares_second_annotate.pdf"),
       height = 4.5, width = 8, unit = "in")




# tibble(
#   year = seq(1, 40, by = 1),
#   pre_pay = pmt_stream(pre_mortg)$pay,
#   cares_pay = cares_recap(pre_mortg)$pay
#   ) %>%
#   transmute(year,
#             d_cares = cares_pay - pre_pay) %>%
#   left_join(df_hamp, by = "year") %>%
#   select(year, d_hamp, d_cares) %>%
#   pivot_longer(cols = -year,
#                values_to = "d_pay",
#                names_to = "mod") %>%
#   mutate(mod = factor(mod, c("d_hamp", "d_cares"))) %>%
#   ggplot(aes(x = year, y = d_pay, fill = mod)) +
#   geom_hline(yintercept = 0,
#              linetype = "dashed",
#              size = 0.25) +
#   geom_vline(xintercept = 23,
#              linetype = "dashed",
#              size = 0.3) +
#   annotate("text", x = 23.1, y = 13000,
#            label = "Original term length of 23 years",
#            family = "serif",
#            hjust = 0, size = 3) +
#   geom_col(position = "dodge") +
#   scale_y_continuous(labels = scales::dollar_format()) +
#   scale_x_continuous(limits = c(0, 41),
#                      expand = c(0, 0)) +
#   scale_fill_manual(values = c("#8ecae6", "#9d0208"),
#                     labels = c("HAMP (Great Recession)",
#                                "CARES (Pandemic)")) +
#   labs(x = "Years since intervention",
#        y = "",
#        subtitle = "Change in annual payment after intervention for typical mortgage",
#        fill = "") +
#   fte_theme() +
#   theme(legend.position = c(0.95, 0.05),
#         legend.justification = c("right", "bottom"),
#         axis.text = element_text(size = 10),
#         legend.text = element_text(size = 12),
#         plot.title.position = "plot")
# 
# ggsave(str_c("issue_22/", "both_mod_capitalization.png"),
#        height = 4.5, width = 8, unit = "in")

tibble(
  year = seq(1, 40, by = 1),
  pre_pay = pmt_stream(pre_mortg)$pay,
  cares_pay = cares_recap(pre_mortg)$pay
) %>%
  transmute(year,
            d_cares = cares_pay - pre_pay) %>%
  left_join(df_hamp, by = "year") %>%
  select(year, d_hamp, d_cares) %>%
  pivot_longer(cols = -year,
               values_to = "d_pay",
               names_to = "mod") %>%
  mutate(mod = factor(mod, c("d_hamp", "d_cares"))) %>%
  ggplot(aes(x = year, y = d_pay, fill = mod)) +
  geom_hline(yintercept = 0,
             linetype = "dashed",
             size = 0.25) +
  geom_vline(xintercept = 23,
             linetype = "dashed",
             size = 0.3) +
  annotate("text", x = 23.1, y = 13000,
           label = "Original term length of 23 years",
           family = "serif",
           hjust = 0, size = 3) +
  geom_col(position = "dodge") +
  annotate("text", x = 1.7, y = -10000,
           label = "Year 1 payment reduction: $15,600 (CARES) vs $5,500 (HAMP)\nNPV total payment reduction: -$2,200 (CARES) vs  $44,800 (HAMP)",
           family = "serif",
           size = 3, hjust = 0) +
  scale_y_continuous(labels = scales::dollar_format()) +
  scale_x_continuous(limits = c(0, 41),
                     expand = c(0, 0)) +
  scale_fill_manual(values = c("#8ecae6", "#9d0208"),
                    labels = c("HAMP (Great Recession)",
                               "CARES (Pandemic)")) +
  labs(x = "Years since intervention",
       y = "",
       subtitle = "Change in annual payment after intervention for typical mortgage",
       fill = "") +
  fte_theme() +
  theme(legend.position = c(0.95, 0.05),
        legend.justification = c("right", "bottom"),
        axis.text = element_text(size = 10),
        legend.text = element_text(size = 12),
        plot.title.position = "plot")

ggsave(str_c("issue_22/", "both_mod_capitalization_annotation.pdf"),
       height = 4.5, width = 8, unit = "in")

df_pay %>%
  rename(d_cares_last_period = d_cares) %>%
  left_join(tibble(
    year = seq(1, 40, by = 1),
    pre_pay = pmt_stream(pre_mortg)$pay,
    cares_pay = pmt_stream_cares(pre_mortg, one_time = "start")$pay
  ) %>%
    transmute(year,
              d_cares_second_period = cares_pay - pre_pay)) %>%
  left_join(tibble(
    year = seq(1, 40, by = 1),
    pre_pay = pmt_stream(pre_mortg)$pay,
    cares_pay = cares_recap(pre_mortg)$pay
  ) %>%
    transmute(year,
              d_cares_capitalization = cares_pay - pre_pay)) %>%
  transmute(Year = year,
            `Payment difference HAMP` = d_hamp,
            `Payment difference CARES - payback last period` = d_cares_last_period,
            `Payment difference CARES - payback second period` = d_cares_second_period,
            `Payment difference CARES - payback capitalization` = d_cares_capitalization) %>%
  write.xlsx(file = str_c("issue_22/", "data.xlsx"),
             sheetName = "Cashflows",
             append = FALSE)



##################
# Summary Impact #
##################

discount_rate <- 0.03

# CARES forbearance in last period
sum_last <- df_pay %>%
  mutate(
    disct_fact = 1 / (1 + discount_rate)^year,
    npv_hamp = d_hamp * disct_fact,
    npv_cares = d_cares * disct_fact) %>%
  mutate_all(~ . * -1) %>%
  summarise_all(sum) %>% 
  transmute(npv_hamp = npv_hamp,
            npv_cares = npv_cares) %>%
  bind_cols(df_pay %>%
              filter(year == 1) %>%
              transmute(year_cares = -d_cares,
                        year_hamp = -d_hamp))

sum_last %>%
  pivot_longer(cols = npv_hamp:year_hamp,
               names_to = "spec",
               values_to = "reduction") %>%
  separate(col = spec, c("acc", "program")) %>%
  mutate(acc = factor(acc, c("year", "npv")),
         program = factor(program, c("hamp", "cares"))) %>%
  ggplot(aes(x = acc, y = reduction, fill = program)) +
  geom_col(position = "dodge", alpha = 0.9) +
  scale_y_continuous(labels = scales::dollar_format()) +
  scale_x_discrete(labels = c("1-year payment\nreduction",
                              "Reduction in NPV of\npayments owed at 3%\ndiscount rate")) +
  scale_fill_manual(values = c("#8ecae6",
                               "#9d0208"),
                     labels = c("HAMP (Great Recession)",
                                "CARES (Pandemic)")) +
  labs(x = "",
       y = "",
       fill = "") +
  fte_theme() +
  theme(legend.justification = c("left", "top"),
        legend.text = element_text(size = 12),
        legend.position = c(0.01, 0.95))


ggsave(str_c("issue_22/", "summary_impact_cares_last.pdf"),
       height = 4.5, width = 8, unit = "in")

sum_last %>%
  pivot_longer(cols = npv_hamp:year_hamp,
               names_to = "spec",
               values_to = "reduction") %>%
  separate(col = spec, c("acc", "program")) %>%
  pivot_wider(id_cols = program,
              names_from = acc,
              values_from = reduction) %>%
  mutate(reduction_cost = npv / year) %>%
  transmute(Program = program,
            `1-year payment reduction` = comma(round(year / 100) * 100),
            `Reduction in NPV of payments owed` = comma(round(npv / 100) * 100),
            `Cost of a dollar immediate relief (in $)` = round(reduction_cost, digits = 2)) %>%
  write.xlsx(file = str_c("issue_22/", "data.xlsx"),
             sheetName = "Summary Impact", append = TRUE)


# 
# # CARES forbearance in second period
# sum_second <- tibble(
#   year = seq(1, 40, by = 1),
#   pre_pay = pmt_stream(pre_mortg)$pay,
#   cares_pay = pmt_stream_cares(pre_mortg, one_time = "start")$pay
# ) %>%
#   transmute(year,
#             d_cares = cares_pay - pre_pay) %>%
#   left_join(df_hamp, by = "year") %>%
#   mutate(
#     disct_fact = 1 / (1 + discount_rate)^year,
#     npv_hamp = d_hamp * disct_fact,
#     npv_cares = d_cares * disct_fact) %>%
#   mutate_all(~ . * -1) %>%
#   summarise_all(sum) %>% 
#   transmute(npv_hamp = npv_hamp,
#             npv_cares = npv_cares) %>%
#   bind_cols(df_pay %>%
#               filter(year == 1) %>%
#               transmute(year_cares = -d_cares,
#                         year_hamp = -d_hamp))
# 
# sum_second %>%
#   pivot_longer(cols = npv_hamp:year_hamp,
#                names_to = "spec",
#                values_to = "reduction") %>%
#   separate(col = spec, c("acc", "program")) %>%
#   mutate(acc = factor(acc, c("year", "npv")),
#          program = factor(program, c("hamp", "cares"))) %>%
#   ggplot(aes(x = acc, y = reduction, fill = program)) +
#   geom_col(position = "dodge", alpha = 0.9) +
#   scale_y_continuous(labels = scales::dollar_format()) +
#   scale_x_discrete(labels = c("1-year payment\nreduction",
#                               "Reduction in NPV of\npayments owed at 3%\ndiscount rate")) +
#   scale_fill_manual(values = c("#8ecae6",
#                                "#9d0208"),
#                     labels = c("HAMP (Great Recession)",
#                                "CARES (Pandemic)")) +
#   labs(x = "",
#        y = "",
#        fill = "") +
#   fte_theme() +
#   theme(legend.justification = c("left", "top"),
#         legend.text = element_text(size = 12),
#         legend.position = c(0.01, 0.95))
# 
# ggsave(str_c("issue_22/", "summary_impact_cares_second.png"),
#        height = 4.5, width = 8, unit = "in")
# 
# sum_second %>%
#   pivot_longer(cols = npv_hamp:year_hamp,
#                names_to = "spec",
#                values_to = "reduction") %>%
#   separate(col = spec, c("acc", "program")) %>%
#   pivot_wider(id_cols = program,
#               names_from = acc,
#               values_from = reduction) %>%
#   mutate(reduction_cost = npv / year) %>%
#   transmute(Program = program,
#             `1-year payment reduction` = comma(round(year / 100) * 100),
#             `Reduction in NPV of payments owed` = comma(round(npv / 100) * 100),
#             `Cost of a dollar immediate relief (in $)` = round(reduction_cost, digits = 2)) %>%
#   knitr::kable()
# 
# 
# # CARES capitalization
# sum_capitalization <- tibble(
#   year = seq(1, 40, by = 1),
#   pre_pay = pmt_stream(pre_mortg)$pay,
#   cares_pay = cares_recap(pre_mortg)$pay
# ) %>%
#   transmute(year,
#             d_cares = cares_pay - pre_pay) %>%
#   left_join(df_hamp, by = "year") %>%
#   mutate(
#     disct_fact = 1 / (1 + discount_rate)^year,
#     npv_hamp = d_hamp * disct_fact,
#     npv_cares = d_cares * disct_fact) %>%
#   mutate_all(~ . * -1) %>%
#   summarise_all(sum) %>% 
#   transmute(npv_hamp = npv_hamp,
#             npv_cares = npv_cares) %>%
#   bind_cols(df_pay %>%
#               filter(year == 1) %>%
#               transmute(year_cares = -d_cares,
#                         year_hamp = -d_hamp))
# 
# sum_capitalization %>%
#   pivot_longer(cols = npv_hamp:year_hamp,
#                names_to = "spec",
#                values_to = "reduction") %>%
#   separate(col = spec, c("acc", "program")) %>%
#   mutate(acc = factor(acc, c("year", "npv")),
#          program = factor(program, c("hamp", "cares"))) %>%
#   ggplot(aes(x = acc, y = reduction, fill = program)) +
#   geom_col(position = "dodge", alpha = 0.9) +
#   scale_y_continuous(labels = scales::dollar_format()) +
#   scale_x_discrete(labels = c("1-year payment\nreduction",
#                               "Reduction in NPV of\npayments owed at 3%\ndiscount rate")) +
#   scale_fill_manual(values = c("#8ecae6",
#                                "#9d0208"),
#                     labels = c("HAMP (Great Recession)",
#                                "CARES (Pandemic)")) +
#   labs(x = "",
#        y = "",
#        fill = "") +
#   fte_theme() +
#   theme(legend.justification = c("left", "top"),
#         legend.text = element_text(size = 12),
#         legend.position = c(0.01, 0.95))
# 
# ggsave(str_c("issue_22/", "summary_impact_cares_capitalization.png"),
#        height = 4.5, width = 8, unit = "in")
# 
# sum_capitalization %>%
#   pivot_longer(cols = npv_hamp:year_hamp,
#                names_to = "spec",
#                values_to = "reduction") %>%
#   separate(col = spec, c("acc", "program")) %>%
#   pivot_wider(id_cols = program,
#               names_from = acc,
#               values_from = reduction) %>%
#   mutate(reduction_cost = npv / year) %>%
#   transmute(Program = program,
#             `1-year payment reduction` = comma(round(year / 100) * 100),
#             `Reduction in NPV of payments owed` = comma(round(npv / 100) * 100),
#             `Cost of a dollar immediate relief (in $)` = round(reduction_cost, digits = 2)) %>%
#   knitr::kable()
# 
